<?php
/**
 * This file is used to generate the "Sliding News" page in the admin.
 *
 * @package live-news
 */

if ( ! current_user_can( get_option( $this->shared->get( 'slug' ) . '_sliding_menu_capability' ) ) ) {
	wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'daln' ) );
}

?>

<!-- process data -->

<?php

// Initialize variables -----------------------------------------------------------------------------------------------.
$dismissible_notice_a = array();

// Preliminary operations ---------------------------------------------------------------------------------------------.
global $wpdb;

// Sanitization -------------------------------------------------------------------------------------------------------.

// Actions.
$data['edit_id']        = isset( $_GET['edit_id'] ) ? intval( $_GET['edit_id'], 10 ) : null;
$data['delete_id']      = isset( $_POST['delete_id'] ) ? intval( $_POST['delete_id'], 10 ) : null;
$data['update_id']      = isset( $_POST['update_id'] ) ? intval( $_POST['update_id'], 10 ) : null;
$data['form_submitted'] = isset( $_POST['form_submitted'] ) ? intval( $_POST['form_submitted'], 10 ) : null;

// Filter and search data.
$data['s']  = isset( $_GET['s'] ) ? sanitize_text_field( $_GET['s'] ) : null;
$data['cf'] = isset( $_GET['cf'] ) ? sanitize_text_field( $_GET['cf'] ) : null;

if ( ! is_null( $data['update_id'] ) || ! is_null( $data['form_submitted'] ) ) {

	// Nonce verification..
	check_admin_referer( 'daln_create_update_sliding_news', 'daln_create_update_sliding_news_nonce' );

	// Sanitization ---------------------------------------------------------------------------------------------------.
	$data['news_title']               = isset( $_POST['news_title'] ) ? sanitize_text_field( $_POST['news_title'] ) : null;
	$data['url']                      = isset( $_POST['url'] ) ? esc_url_raw( $_POST['url'] ) : null;
	$data['ticker_id']                = isset( $_POST['ticker_id'] ) ? intval( $_POST['ticker_id'], 10 ) : null;
	$data['text_color']               = isset( $_POST['text_color'] ) ? sanitize_text_field( $_POST['text_color'] ) : null;
	$data['text_color_hover']         = isset( $_POST['text_color_hover'] ) ? sanitize_text_field( $_POST['text_color_hover'] ) : null;
	$data['background_color']         = isset( $_POST['background_color'] ) ? sanitize_text_field( $_POST['background_color'] ) : null;
	$data['background_color_opacity'] = isset( $_POST['background_color_opacity'] ) ? floatval( $_POST['background_color_opacity'] ) : null;
	$data['image_before']             = isset( $_POST['image_before'] ) ? esc_url_raw( $_POST['image_before'] ) : null;
	$data['image_after']              = isset( $_POST['image_after'] ) ? esc_url_raw( $_POST['image_after'] ) : null;

	// Validation -----------------------------------------------------------------------------------------------------.

	// Validation on "Title".
	if ( mb_strlen( trim( $data['news_title'] ) ) === 0 || mb_strlen( $data['news_title'] ) > 1000 ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid value in the "Title" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "URL".
	if ( mb_strlen( $data['url'] ) > 2083 ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid URL in the "URL" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "Text Color".
	if ( ! preg_match( $this->shared->hex_rgb_regex, $data['text_color'] ) ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid color in the "Text Color" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "Text Color Hover".
	if ( ! preg_match( $this->shared->hex_rgb_regex, $data['text_color_hover'] ) ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid color in the "Text Color Hover" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "Background Color".
	if ( ! preg_match( $this->shared->hex_rgb_regex, $data['background_color'] ) ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid color in the "Background Color" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "Background Color Opacity".
	if ( $data['background_color_opacity'] < 0 || $data['background_color_opacity'] > 1 ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a value included between 0 and 1 in the "Background Color Opacity" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "Image Before".
	if ( mb_strlen( $data['image_before'] ) > 2083 ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid URL in the "Image Left" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}

	// Validation on "Image After".
	if ( mb_strlen( $data['image_after'] ) > 2083 ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'Please enter a valid URL in the "Image Right" field.', 'daln' ),
			'class'   => 'error',
		);
		$invalid_data           = true;
	}
}

// Update ---------------------------------------------------------------.
if ( ! is_null( $data['update_id'] ) && ! isset( $invalid_data ) ) {

	// Update the database.
	$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
	$safe_sql   = $wpdb->prepare(
		"UPDATE $table_name SET
                news_title = %s,
                url = %s,
                ticker_id = %d,
                text_color = %s,
                text_color_hover = %s,
                background_color = %s,
                background_color_opacity = %f,
                image_before = %s,
                image_after = %s
                WHERE id = %d",
		$data['news_title'],
		$data['url'],
		$data['ticker_id'],
		$data['text_color'],
		$data['text_color_hover'],
		$data['background_color'],
		$data['background_color_opacity'],
		$data['image_before'],
		$data['image_after'],
		$data['update_id']
	);

	$query_result = $wpdb->query( $safe_sql );

	if ( false !== $query_result ) {
		$dismissible_notice_a[] = array(
			'message' => __( 'The sliding news has been successfully updated.', 'daln' ),
			'class'   => 'updated',
		);
	}
} else {

	// Add ------------------------------------------------------------------.
	if ( ! is_null( $data['form_submitted'] ) && ! isset( $invalid_data ) ) {

		// Insert into the database.
		$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
		$safe_sql   = $wpdb->prepare(
			"INSERT INTO $table_name SET
                    news_title = %s,
                    url = %s,
                    ticker_id = %d,
                    text_color = %s,
                    text_color_hover = %s,
                    background_color = %s,
                    background_color_opacity = %f,
                    image_before = %s,
                    image_after = %s",
			$data['news_title'],
			$data['url'],
			$data['ticker_id'],
			$data['text_color'],
			$data['text_color_hover'],
			$data['background_color'],
			$data['background_color_opacity'],
			$data['image_before'],
			$data['image_after']
		);

		$query_result = $wpdb->query( $safe_sql );

		if ( false !== $query_result ) {
			$dismissible_notice_a[] = array(
				'message' => __( 'The sliding news has been successfully added.', 'daln' ),
				'class'   => 'updated',
			);
		}
	}
}

// Delete a sliding news.
if ( ! is_null( $data['delete_id'] ) ) {

	// Nonce verification.
	check_admin_referer( 'daln_delete_sliding_news_' . $data['delete_id'], 'daln_delete_sliding_news_nonce' );

	// Delete this game.
	$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
	$safe_sql   = $wpdb->prepare( "DELETE FROM $table_name WHERE id = %d ", $data['delete_id'] );

	$query_result = $wpdb->query( $safe_sql );

	if ( false !== $query_result) {
		$dismissible_notice_a[] = array(
			'message' => __( 'The sliding news has been successfully deleted.', 'daln' ),
			'class'   => 'updated',
		);
	}
}

// Get the sliding news data.
$display_form = true;
if ( ! is_null( $data['edit_id'] ) ) {
	$table_name       = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
	$safe_sql         = $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d ", $data['edit_id'] );
	$sliding_news_obj = $wpdb->get_row( $safe_sql );
	if ( null === $sliding_news_obj ) {
		$display_form = false;
	}
}

// Get the value of the custom filter.
if ( null !== $data['cf'] ) {
	if ( 'all' !== $data['cf'] ) {
		$ticker_id_in_cf = intval( $data['cf'], 10 );
	} else {
		$ticker_id_in_cf = false;
	}
} else {
	$ticker_id_in_cf = false;
}

?>

<!-- output -->

<div class="wrap">

	<?php if ( $this->shared->get_number_of_sliding_news() > 0 ) : ?>

		<div id="daext-header-wrapper">

			<h2><?php esc_html_e( 'Live News - Sliding News', 'daln' ); ?></h2>

			<div class="daext-header-forms">

				<!-- Filter Form -->

				<form method="GET" action="admin.php" id="daext-filter-form">

					<input type="hidden" name="page" value="<?php echo $this->shared->get( 'slug' ); ?>-sliding">

					<p><?php esc_html_e( 'Filter by News Ticker', 'daln' ); ?></p>

					<select id="cf" name="cf" class="daext-display-none">

						<option value="all" 
						<?php
						if ( null !== $data['cf'] ) {
							selected( $data['cf'], 'all' );
						}
						?>
						><?php esc_html_e( 'All', 'daln' ); ?></option>

						<?php

						$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_tickers';
						$safe_sql   = "SELECT id, name FROM $table_name ORDER BY id DESC";
						$tickers_a  = $wpdb->get_results( $safe_sql, ARRAY_A );

						foreach ( $tickers_a as $key => $ticker ) {

							if ( null !== $data['cf'] ) {
								echo '<option value="' . esc_attr( $ticker['id'] ) . '" ' . selected( $data['cf'], $ticker['id'], false ) . '>' . esc_html( stripslashes( $ticker['name'] ) ) . '</option>';
							} else {
								echo '<option value="' . esc_attr( $ticker['id'] ) . '">' . esc_html( stripslashes( $ticker['name'] ) ) . '</option>';

							}
						}

						?>

					</select>

				</form>

				<!-- Search Form -->

				<form action="admin.php" method="get" id="daext-search-form">

					<input type="hidden" name="page" value="daln-sliding">

					<p><?php esc_html_e( 'Perform your Search', 'daln' ); ?></p>

					<?php
					if ( ! is_null( $data['s'] ) ) {
						if ( mb_strlen( trim( $data['s'] ) ) > 0 ) {
							$search_string = $data['s'];
						} else {
							$search_string = '';
						}
					} else {
						$search_string = '';
					}

					// Custom Filter.
					if ( false !== $ticker_id_in_cf ) {
						echo '<input type="hidden" name="cf" value="' . esc_attr( $ticker_id_in_cf ) . '">';
					}

					?>

					<input type="text" name="s" name="s"
							value="<?php echo esc_attr( stripslashes( $search_string ) ); ?>" autocomplete="off"
							maxlength="255">
					<input type="submit" value="">

				</form>

			</div>

		</div>

	<?php else : ?>

		<div id="daext-header-wrapper" class="daext-clearfix">

			<h2><?php esc_html_e( 'Live News - Sliding News', 'daln' ); ?></h2>

		</div>

	<?php endif; ?>

	<?php

	// Do not display the menu if in the 'cf' url parameter is applied a filter based on a ticker that doesn't exist.
	if ( null !== $data['cf'] ) {
		if ( 'all' !== $data['cf'] && ! $this->shared->ticker_exists( $data['cf'] ) ) {
			echo '<p>' . esc_html__( "The filter can't be applied because this news ticker doesn't exist.", 'daln' ) . '</p>';

			return;
		}
	}

	// Retrieve the url parameter that should be used in the linked URLs.
	if ( $this->shared->ticker_exists( $data['cf'] ) ) {
		$ticker_url_parameter = '&cf=' . intval( $data['cf'], 10 );
	} else {
		$ticker_url_parameter = '';
	}

	// Display a message and not the menu if there are no tickers.
	if ( 0 === $this->shared->get_number_of_tickers() ) {
		echo '<p>' . esc_html__( 'There are no news tickers at the moment, please create at least one news ticker with the', 'daln' ) . ' ' . '<a href="admin.php?page=daln-tickers">' . esc_html__( 'News Tickers', 'daln' ) . '</a> menu.' . '</p>';

		return;
	}

	?>

	<div id="daext-menu-wrapper">

		<?php $this->dismissible_notice( $dismissible_notice_a ); ?>

		<!-- table -->

		<?php

		// Custom filter.
		if ( false === $ticker_id_in_cf ) {
			$filter = '';
		} else {
			$filter = $wpdb->prepare( 'WHERE ticker_id = %d', $ticker_id_in_cf );
		}

		// Create the query part used to filter the results when a search is performed.
		if ( ! is_null( $data['s'] ) ) {

			if ( mb_strlen( trim( $data['s'] ) ) > 0 ) {

				if ( strlen( trim( $filter ) ) > 0 ) {
					$filter .= $wpdb->prepare( ' AND (news_title LIKE %s OR url LIKE %s)', '%' . $search_string . '%', '%' . $search_string . '%' );
				} else {
					$filter = $wpdb->prepare( 'WHERE (news_title LIKE %s OR url LIKE %s)', '%' . $search_string . '%', '%' . $search_string . '%' );
				}
			}
		}

		// retrieve the total number of sliding news.
		$table_name  = $wpdb->prefix . $this->shared->get( 'slug' ) . '_sliding_news';
		$total_items = $wpdb->get_var( "SELECT COUNT(*) FROM $table_name $filter" );

		// Initialize the pagination class.
		require_once $this->shared->get( 'dir' ) . '/admin/inc/class-daln-pagination.php';
		$pag = new daln_pagination();
		$pag->set_total_items( $total_items );// Set the total number of items.
		$pag->set_record_per_page( 10 ); // Set records per page.
		$pag->set_target_page( 'admin.php?page=' . $this->shared->get( 'slug' ) . '-sliding' );// Set target page.
		$pag->set_current_page();// Set the current page number.

		?>

		<!-- Query the database -->
		<?php
		$query_limit = $pag->query_limit();
		$results     = $wpdb->get_results( "SELECT * FROM $table_name $filter ORDER BY id DESC $query_limit ", ARRAY_A );
		?>

		<?php if ( count( $results ) > 0 ) : ?>

			<div class="daext-items-container">

				<!-- list of tables -->
				<table class="daext-items">
					<thead>
					<tr>
						<th>
							<div class="daext-items-th-inner">
								<div><?php esc_html_e( 'Title', 'daln' ); ?></div>
								<div class="help-icon"
									title="<?php esc_attr_e( 'The title of the sliding news.', 'daln' ); ?>"></div>
							</div>
						</th>
						<th>
							<div class="daext-items-th-inner">
								<div><?php esc_html_e( 'Ticker', 'daln' ); ?></div>
								<div class="help-icon"
									title="<?php esc_attr_e( 'The news ticker associated with the sliding news.', 'daln' ); ?>"></div>
							</div>
						</th>
						<th></th>
					</tr>
					</thead>
					<tbody>

					<?php foreach ( $results as $result ) : ?>
						<tr>
							<td><?php echo esc_html( stripslashes( $result['news_title'] ) ); ?></td>
							<td><?php echo esc_html( stripslashes( $this->shared->get_textual_ticker( $result['ticker_id'] ) ) ); ?></td>
							<td class="icons-container">
								<div class="icons-container-inner">
									<a class="menu-icon edit"
										href="admin.php?page=<?php echo esc_attr( $this->shared->get( 'slug' ) ); ?>-sliding&edit_id=<?php echo esc_attr( $result['id'] ); ?><?php echo esc_attr( $ticker_url_parameter ); ?>"></a>
									<form method="POST"
											action="admin.php?page=<?php echo esc_attr( $this->shared->get( 'slug' ) ); ?>-sliding">
										<?php wp_nonce_field( 'daln_delete_sliding_news_' . $result['id'], 'daln_delete_sliding_news_nonce' ); ?>
										<input type="hidden" value="<?php echo esc_attr( $result['id'] ); ?>"
												name="delete_id">
										<input class="menu-icon delete" type="submit" value="">
									</form>
								</div>
							</td>
						</tr>
					<?php endforeach; ?>

					</tbody>

				</table>

			</div>

			<!-- Display the pagination -->
			<?php if ( $pag->total_items > 0 ) : ?>
				<div class="daext-tablenav daext-clearfix">
					<div class="daext-tablenav-pages">
						<span class="daext-displaying-num"><?php echo esc_html( $pag->total_items ); ?>&nbsp;<?php esc_html_e( 'items', 'daln' ); ?></span>
						<?php $pag->show(); ?>
					</div>
				</div>
			<?php endif; ?>

		<?php else : ?>

			<?php

			if ( strlen( trim( $filter ) ) > 0 ) {
				echo '<div class="error settings-error notice is-dismissible below-h2"><p>' . esc_html__( 'There are no results that match your filter.', 'daln' ) . '</p></div>';
			}

			?>

		<?php endif; ?>

		<div id="sliding-news-form-container">

			<?php if ( $display_form ) : ?>

				<form method="POST"
						action="admin.php?page=<?php echo esc_attr( $this->shared->get( 'slug' ) ); ?>-sliding<?php echo esc_html( $ticker_url_parameter ); ?>"
						autocomplete="off">

					<input type="hidden" value="1" name="form_submitted">
					<?php wp_nonce_field( 'daln_create_update_sliding_news', 'daln_create_update_sliding_news_nonce' ); ?>

					<?php if ( ! is_null( $data['edit_id'] ) ) : ?>

					<!-- Edit a sliding news -->

					<div class="daext-form-container">

						<h3 class="daext-form-title"><?php esc_html_e( 'Edit Sliding News', 'daln' ); ?><?php echo esc_html( $sliding_news_obj->id ); ?></h3>

						<table class="daext-form">

							<input type="hidden" name="update_id" id="update-id"
									value="<?php echo esc_attr( $sliding_news_obj->id ); ?>"/>

							<!-- title -->
							<tr valign="top">
								<th scope="row"><label for="news-title"><?php esc_html_e( 'Title', 'daln' ); ?></label>
								</th>
								<td>
									<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->news_title ) ); ?>"
											type="text" id="news-title" maxlength="1000" size="30" name="news_title"/>
									<div class="help-icon"
										title="<?php esc_attr_e( 'Enter the title of the sliding news.', 'daln' ); ?>"></div>
								</td>
							</tr>

							<!-- URL -->
							<tr valign="top">
								<th scope="row"><label for="url"><?php esc_html_e( 'URL', 'daln' ); ?></label></th>
								<td>
									<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->url ) ); ?>"
											type="text" id="url" maxlength="2083" size="30" name="url"/>
									<div class="help-icon"
										title="<?php esc_attr_e( 'Enter the URL of the sliding news.', 'daln' ); ?>"></div>
								</td>
							</tr>

							<!-- Ticker -->
							<tr>
								<th scope="row"><?php esc_html_e( 'Ticker', 'daln' ); ?></th>
								<td>
									<select id="ticker-id" name="ticker_id" class="daext-display-none">

										<?php

										$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_tickers';
										$safe_sql   = "SELECT id, name FROM $table_name ORDER BY id DESC";
										$tickers_a  = $wpdb->get_results( $safe_sql, ARRAY_A );

										foreach ( $tickers_a as $key => $ticker ) {

											echo '<option value="' . esc_attr( $ticker['id'] ) . '" ' . selected( $sliding_news_obj->ticker_id, $ticker['id'] ) . '>' . esc_html( stripslashes( $ticker['name'] ) ) . '</option>';

										}

										?>

									</select>
									<div class="help-icon"
										title='<?php esc_attr_e( 'Select the news ticker associated with this sliding news.', 'daln' ); ?>'></div>
								</td>
							</tr>

							<!-- Text Color -->
							<tr valign="top">
								<th scope="row"><label
											for="text-color"><?php esc_html_e( 'Text Color', 'daln' ); ?></label></th>
								<td>
									<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->text_color ) ); ?>"
											class="wp-color-picker" type="text" id="text-color" maxlength="7" size="30"
											name="text_color"/>
									<div class="help-icon"
										title="<?php esc_attr_e( 'Select the color used to display the text of this sliding news.', 'daln' ); ?>"></div>
								</td>
							</tr>

							<!-- Text Color Hover -->
							<tr valign="top">
								<th scope="row"><label
											for="text-color-hover"><?php esc_html_e( 'Text Color Hover', 'daln' ); ?></label>
								</th>
								<td>
									<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->text_color_hover ) ); ?>"
											class="wp-color-picker" type="text" id="text-color-hover" maxlength="7"
											size="30" name="text_color_hover"/>
									<div class="help-icon"
										title="<?php esc_attr_e( 'Select the color used to display the text of this sliding news in hover state.', 'daln' ); ?>"></div>
								</td>
							</tr>

							<!-- Background Color -->
							<tr valign="top">
								<th scope="row"><label
											for="background-color"><?php esc_html_e( 'Background Color', 'daln' ); ?></label>
								</th>
								<td>
									<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->background_color ) ); ?>"
											class="wp-color-picker" type="text" id="background-color" maxlength="7"
											size="30" name="background_color"/>
									<div class="help-icon"
										title="<?php esc_attr_e( 'Select the background color of this sliding news.', 'daln' ); ?>"></div>
								</td>
							</tr>

							<!-- Background Color Opacity -->
							<tr>
								<th scope="row"><label
											for="background-color-opacity"><?php esc_html_e( 'Background Color Opacity', 'daln' ); ?></label>
								</th>
								<td>
									<input value="<?php echo floatval( $sliding_news_obj->background_color_opacity ); ?>"
											type="text" id="background-color-opacity" maxlength="3" size="30"
											name="background_color_opacity"/>
									<div class="help-icon"
										title="<?php esc_attr_e( 'The background color opacity of this sliding news.', 'daln' ); ?>"></div>
								</td>
							</tr>

							<!-- Image Before -->
							<tr>
								<th scope="row"><label
											for="image-before"><?php esc_html_e( 'Image Left', 'daln' ); ?></label></th>
								<td>

									<div class="image-uploader">
										<img class="selected-image"
											src="<?php echo esc_attr( stripslashes( $sliding_news_obj->image_before ) ); ?>" <?php echo mb_strlen( trim( $sliding_news_obj->image_before ) ) === 0 ? 'style="display: none;"' : ''; ?>>
										<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->image_before ) ); ?>"
												type="hidden" id="image-before" maxlength="2083" name="image_before">
										<a class="button_add_media"
											data-set-remove="<?php echo mb_strlen( trim( $sliding_news_obj->image_before ) ) === 0 ? 'set' : 'remove'; ?>"
											data-set="<?php esc_attr_e( 'Set image', 'daln' ); ?>"
											data-remove="<?php esc_attr_e( 'Remove Image', 'daln' ); ?>"><?php echo mb_strlen( trim( $sliding_news_obj->image_before ) ) === 0 ? esc_attr__( 'Set image', 'daln' ) : esc_attr__( 'Remove Image', 'daln' ); ?></a>
										<p class="description"><?php esc_html_e( "Select the image displayed on the left of the sliding news. It's recommended to use an image with an height of 40 pixels.", 'daln' ); ?></p>
									</div>

								</td>
							</tr>

							<!-- Image After -->
							<tr>
								<th scope="row"><label
											for="image-after"><?php esc_html_e( 'Image Right', 'daln' ); ?></label></th>
								<td>

									<div class="image-uploader">
										<img class="selected-image"
											src="<?php echo esc_attr( stripslashes( $sliding_news_obj->image_after ) ); ?>" <?php echo mb_strlen( trim( $sliding_news_obj->image_after ) ) === 0 ? 'style="display: none;"' : ''; ?>>
										<input value="<?php echo esc_attr( stripslashes( $sliding_news_obj->image_after ) ); ?>"
												type="hidden" id="image-after" maxlength="2083" name="image_after">
										<a class="button_add_media"
											data-set-remove="<?php echo mb_strlen( trim( $sliding_news_obj->image_after ) ) === 0 ? 'set' : 'remove'; ?>"
											data-set="<?php esc_attr_e( 'Set image', 'daln' ); ?>"
											data-remove="<?php esc_attr_e( 'Remove Image', 'daln' ); ?>"><?php echo mb_strlen( trim( $sliding_news_obj->image_after ) ) === 0 ? esc_attr__( 'Set image', 'daln' ) : esc_attr__( 'Remove Image', 'daln' ); ?></a>
										<p class="description"><?php esc_html_e( "Select the image displayed on the right of the sliding news. It's recommended to use an image with an height of 40 pixels.", 'daln' ); ?></p>
									</div>

								</td>
							</tr>

						</table>

						<!-- submit button -->
						<div class="daext-form-action">
							<input class="button" type="submit"
									value="<?php esc_attr_e( 'Update Sliding News', 'daln' ); ?>">
						</div>

						<?php else : ?>

						<!-- Create New Sliding News -->

						<div class="daext-form-container">

							<div class="daext-form-title"><?php esc_html_e( 'Create a Sliding News', 'daln' ); ?></div>

							<table class="daext-form">

								<!-- Title -->
								<tr valign="top">
									<th scope="row"><label
												for="news-title"><?php esc_html_e( 'Title', 'daln' ); ?></label></th>
									<td>
										<input type="text" id="news-title" maxlength="1000" size="30"
												name="news_title"/>
										<div class="help-icon"
											title="<?php esc_attr_e( 'Enter the title of the sliding news.', 'daln' ); ?>"></div>
									</td>
								</tr>

								<!-- URL -->
								<tr valign="top">
									<th scope="row"><label for="url"><?php esc_html_e( 'URL', 'daln' ); ?></label></th>
									<td>
										<input type="text" id="url" maxlength="2083" size="30" name="url"/>
										<div class="help-icon"
											title="<?php esc_attr_e( 'Enter the URL of the sliding news.', 'daln' ); ?>"></div>
									</td>
								</tr>

								<!-- Ticker -->
								<tr>
									<th scope="row"><?php esc_html_e( 'Ticker', 'daln' ); ?></th>
									<td>
										<select id="ticker-id" name="ticker_id" class="daext-display-none">

											<?php

											$table_name = $wpdb->prefix . $this->shared->get( 'slug' ) . '_tickers';
											$safe_sql   = "SELECT id, name FROM $table_name ORDER BY id DESC";
											$tickers_a  = $wpdb->get_results( $safe_sql, ARRAY_A );

											if ( false === $ticker_id_in_cf ) {

												foreach ( $tickers_a as $key => $ticker ) {
													echo '<option value="' . esc_attr( $ticker['id'] ) . '">' . esc_html( stripslashes( $ticker['name'] ) ) . '</option>';
												}
											} else {

												foreach ( $tickers_a as $key => $ticker ) {
													echo '<option value="' . esc_attr( $ticker['id'] ) . '" ' . selected( $ticker_id_in_cf, $ticker['id'], false ) . '>' . esc_html( stripslashes( $ticker['name'] ) ) . '</option>';
												}
											}

											?>

										</select>
										<div class="help-icon"
											title='<?php esc_attr_e( 'Select the news ticker associated with this sliding news.', 'daln' ); ?>'></div>
									</td>
								</tr>

								<!-- Text Color -->
								<tr valign="top">
									<th scope="row"><label
												for="text-color"><?php esc_html_e( 'Text Color', 'daln' ); ?></label>
									</th>
									<td>
										<input class="wp-color-picker" type="text" id="text-color" maxlength="7"
												size="30" name="text_color"/>
										<div class="help-icon"
											title="<?php esc_attr_e( 'Select the color used to display the text of this sliding news.', 'daln' ); ?>"></div>
									</td>
								</tr>

								<!-- Text Color Hover -->
								<tr valign="top">
									<th scope="row"><label
												for="text-color-hover"><?php esc_html_e( 'Text Color Hover', 'daln' ); ?></label>
									</th>
									<td>
										<input class="wp-color-picker" type="text" id="text-color-hover" maxlength="7"
												size="30" name="text_color_hover"/>
										<div class="help-icon"
											title="<?php esc_attr_e( 'Select the color used to display the text of this sliding news in hover state.', 'daln' ); ?>"></div>
									</td>
								</tr>

								<!-- Background Color -->
								<tr valign="top">
									<th scope="row"><label
												for="background-color"><?php esc_html_e( 'Background Color', 'daln' ); ?></label>
									</th>
									<td>
										<input class="wp-color-picker" type="text" id="background-color" maxlength="7"
												size="30" name="background_color"/>
										<div class="help-icon"
											title="<?php esc_attr_e( 'Select the background color of this sliding news.', 'daln' ); ?>"></div>
									</td>
								</tr>

								<!-- Background Color Opacity -->
								<tr>
									<th scope="row"><label
												for="background-color-opacity"><?php esc_html_e( 'Background Color Opacity', 'daln' ); ?></label>
									</th>
									<td>
										<input value="1" type="text" id="background-color-opacity" maxlength="3"
												size="30" name="background_color_opacity"/>
										<div class="help-icon"
											title="<?php esc_attr_e( 'The background color opacity of this sliding news.', 'daln' ); ?>"></div>
									</td>
								</tr>

								<!-- Image Before -->
								<tr>
									<th scope="row"><label
												for="image-before"><?php esc_html_e( 'Image Left', 'daln' ); ?></label>
									</th>
									<td>

										<div class="image-uploader">
											<img class="selected-image" src="" style="display: none">
											<input type="hidden" id="image-before" maxlength="2083" name="image_before">
											<a class="button_add_media" data-set-remove="set"
												data-set="<?php esc_attr_e( 'Set image', 'daln' ); ?>"
												data-remove="<?php esc_attr_e( 'Remove Image', 'daln' ); ?>"><?php esc_html_e( 'Set image', 'daln' ); ?></a>
											<p class="description"><?php esc_html_e( "Select the image displayed on the left of the sliding news. It's recommended to use an image with an height of 40 pixels.", 'daln' ); ?></p>
										</div>

									</td>
								</tr>

								<!-- Image After -->
								<tr>
									<th scope="row"><label
												for="image-after"><?php esc_html_e( 'Image Right', 'daln' ); ?></label>
									</th>
									<td>

										<div class="image-uploader">
											<img class="selected-image" src="" style="display: none">
											<input type="hidden" id="image-after" maxlength="2083" name="image_after">
											<a class="button_add_media" data-set-remove="set"
												data-set="<?php esc_attr_e( 'Set image', 'daln' ); ?>"
												data-remove="<?php esc_attr_e( 'Remove Image', 'daln' ); ?>"><?php esc_html_e( 'Set image', 'daln' ); ?></a>
											<p class="description"><?php esc_attr_e( "Select the image displayed on the right of the sliding news. It's recommended to use an image with an height of 40 pixels.", 'daln' ); ?></p>
										</div>

									</td>
								</tr>

							</table>

							<!-- submit button -->
							<div class="daext-form-action">
								<input class="button" type="submit"
										value="<?php esc_attr_e( 'Add Sliding News', 'daln' ); ?>">
							</div>

							<?php endif; ?>

						</div>

				</form>

			<?php endif; ?>

		</div>

	</div>

</div>